<?php
// core/wp_post.php V4
// Tidak upload media ke WP. Hanya inject <img src="..."> ke konten.
// Plugin Coindev Auto Save Images akan:
// - download gambar
// - simpan ke uploads
// - replace URL
// - set featured image jika belum ada.

require_once __DIR__ . '/../config/database.php';
require_once __DIR__ . '/helper.php';

/**
 * Posting artikel ke WordPress via REST API.
 */
function wp_post_article($template, $article) {
    $wpDomain   = rtrim($template['wp_domain'], '/');
    $wpUser     = $template['wp_user'];
    $wpPass     = $template['wp_app_pass'];
    $categoryId = (int) $template['wp_category'];

    $title   = $article['title'];
    $content = $article['rewritten'] ?: $article['content'];

    // SISIPKAN GAMBAR EXTERNAL DI AWAL KONTEN -> plugin Coindev yang tangani
    if (!empty($article['img_url'])) {
        $imgUrl = $article['img_url'];

        // Paksa format jpeg agar WordPress + plugin lebih mudah terima
        $imgUrl = preg_replace('/format=\w+/i', 'format=jpeg', $imgUrl);
        if (stripos($imgUrl, 'format=') === false) {
            $glue  = (strpos($imgUrl, '?') !== false) ? '&' : '?';
            $imgUrl .= $glue . 'format=jpeg';
        }

        $imgUrlSafe = htmlspecialchars($imgUrl, ENT_QUOTES, 'UTF-8');
        $titleSafe  = htmlspecialchars($title, ENT_QUOTES, 'UTF-8');

        $imgHtml = '<p><img src="' . $imgUrlSafe . '" alt="' . $titleSafe . '" /></p>';

        $content = $imgHtml . "\n\n" . $content;
    }

    $payload = [
        'title'   => $title,
        'content' => $content,
        'status'  => 'publish',
    ];

    if ($categoryId > 0) {
        $payload['categories'] = [$categoryId];
    }

    $headers = [
        'Authorization: Basic ' . base64_encode($wpUser . ':' . $wpPass),
        'Content-Type: application/json',
    ];

    $url = $wpDomain . '/wp-json/wp/v2/posts';

    $response = http_post($url, json_encode($payload), $headers);
    $json     = json_decode($response, true);

    if (!isset($json['id'])) {
        throw new Exception('Gagal membuat post. Response: ' . $response);
    }

    return $json['id'];
}

/**
 * Proses batch artikel: ambil yang status = 'processed' dan kirim ke WordPress.
 */
function process_post_batch($templateId, $limit = 3) {
    global $pdo;

    $stmtTemplate = $pdo->prepare("SELECT * FROM templates WHERE id = ?");
    $stmtTemplate->execute([$templateId]);
    $template = $stmtTemplate->fetch();
    if (!$template) {
        throw new Exception('Template tidak ditemukan (ID: ' . $templateId . ').');
    }

    $stmt = $pdo->prepare("SELECT * FROM articles WHERE status = 'processed' ORDER BY id ASC LIMIT ?");
    $stmt->bindValue(1, (int) $limit, PDO::PARAM_INT);
    $stmt->execute();
    $articles = $stmt->fetchAll();

    foreach ($articles as $article) {
        try {
            echo "Posting artikel ID {$article['id']} ke WordPress...\n";
            $wpPostId = wp_post_article($template, $article);

            $upd = $pdo->prepare("UPDATE articles SET status = 'posted' WHERE id = ?");
            $upd->execute([$article['id']]);

            echo "OK: Artikel ID {$article['id']} berhasil dipost (WP ID: {$wpPostId}).\n";
        } catch (Exception $e) {
            echo "GAGAL posting ID {$article['id']}: " . $e->getMessage() . "\n";
        }
    }
}
