<?php
require_once __DIR__ . '/../config/database.php';
require_once __DIR__ . '/helper.php';

/**
 * Panggil Gemini API untuk rewrite artikel dengan prompt dari template.
 */
function rewrite_with_gemini($template, $originalTitle, $originalContent) {
    $apiKey = $template['api_key'];
    $model  = $template['model'];

    if (!$apiKey || !$model) {
        throw new Exception('API Key atau Model Gemini belum di-set di Template.');
    }

    $endpoint = "https://generativelanguage.googleapis.com/v1beta/models/{$model}:generateContent?key={$apiKey}";

    $basePrompt = trim($template['rewrite_prompt'] ?? '');
    if ($basePrompt === '') {
        $basePrompt = "You are a professional football news writer. Rewrite the article in clear and natural Indonesian journalistic style, keep all important facts, and ensure high uniqueness.";
    }

    $prompt = $basePrompt . "\n\n" .
        "Source title:\n" . $originalTitle . "\n\n" .
        "Source article:\n" . $originalContent . "\n";

    $payload = [
        'contents' => [
            [
                'parts' => [
                    ['text' => $prompt]
                ]
            ]
        ],
        'generationConfig' => [
            'temperature' => floatval($template['temperature']),
            'maxOutputTokens' => intval($template['max_tokens']),
            'topP' => floatval($template['top_p']),
            'topK' => intval($template['top_k']),
        ]
    ];

    $headers = ['Content-Type: application/json'];

    $response = http_post($endpoint, json_encode($payload), $headers);
    $json = json_decode($response, true);

    if (!isset($json['candidates'][0]['content']['parts'][0]['text'])) {
        throw new Exception('Response Gemini tidak sesuai format. Respon mentah: ' . $response);
    }

    $text = trim($json['candidates'][0]['content']['parts'][0]['text']);

    // Jika memakai pattern [TITLE] ... [TITLE]
    if (preg_match('/\[TITLE\](.*)\[TITLE\]/is', $text, $m)) {
        $inner = trim($m[1]);
    } else {
        $inner = $text;
    }

    $title   = $originalTitle;
    $content = $inner;

    $marker = '<strong>NextSport Transfers - </strong>';
    $pos = stripos($inner, $marker);
    if ($pos !== false) {
        $titlePart   = trim(substr($inner, 0, $pos));
        $contentPart = substr($inner, $pos);
        if ($titlePart !== '') {
            $title = preg_replace('/\s+/', ' ', strip_tags($titlePart));
        }
        $content = trim($contentPart);
    } else {
        // fallback: baris pertama = judul, sisanya isi
        $lines = preg_split("/\r?\n/", $inner, 2);
        if (count($lines) === 2) {
            $maybeTitle = trim($lines[0]);
            $maybeBody  = trim($lines[1]);
            if ($maybeTitle !== '') {
                $title = preg_replace('/\s+/', ' ', strip_tags($maybeTitle));
            }
            if ($maybeBody !== '') {
                $content = $maybeBody;
            }
        }
    }

    // Bersihkan sisa [TITLE]
    $title   = str_replace(['[TITLE]','[/TITLE]'], '', $title);
    $content = str_replace(['[TITLE]','[/TITLE]'], '', $content);

    // Hilangkan "| OneFootball" dari judul
    $title = preg_replace('/\|\s*OneFootball/i', '', $title);
    $title = trim($title);

    // Hilangkan markdown bold double-asterisk
    $content = str_replace('**', '', $content);

    return [
        'title'   => $title,
        'content' => trim($content),
    ];
}

/**
 * Proses batch artikel: ambil yang status = 'draft', rewrite, lalu simpan jadi 'processed'.
 */
function process_rewrite_batch($templateId, $limit = 3) {
    global $pdo;

    $stmtTemplate = $pdo->prepare("SELECT * FROM templates WHERE id = ?");
    $stmtTemplate->execute([$templateId]);
    $template = $stmtTemplate->fetch();
    if (!$template) {
        throw new Exception('Template tidak ditemukan (ID: ' . $templateId . ').');
    }

    $stmt = $pdo->prepare("SELECT * FROM articles WHERE status = 'draft' ORDER BY id ASC LIMIT ?");
    $stmt->bindValue(1, (int)$limit, PDO::PARAM_INT);
    $stmt->execute();
    $articles = $stmt->fetchAll();

    foreach ($articles as $article) {
        try {
            echo "Rewriting article ID {$article['id']}...\n";
            $rew = rewrite_with_gemini($template, $article['title'], $article['content']);

            $upd = $pdo->prepare("UPDATE articles SET title = ?, rewritten = ?, status = 'processed' WHERE id = ?");
            $upd->execute([$rew['title'], $rew['content'], $article['id']]);

            echo "OK: ID {$article['id']} berhasil di-rewrite.\n";
        } catch (Exception $e) {
            echo "GAGAL rewrite ID {$article['id']}: " . $e->getMessage() . "\n";
        }
    }
}
