<?php
require_once __DIR__ . '/../config/database.php';
require_once __DIR__ . '/helper.php';

/**
 * Panggil Gemini API untuk rewrite artikel dengan gaya jurnalistik.
 */
function rewrite_with_gemini($template, $originalTitle, $originalContent) {
    $apiKey = $template['api_key'];
    $model = $template['model'];

    if (!$apiKey || !$model) {
        throw new Exception('API Key atau Model Gemini belum di-set di Template.');
    }

    $endpoint = "https://generativelanguage.googleapis.com/v1beta/models/{$model}:generateContent?key={$apiKey}";

    $prompt = "Tulis ulang artikel sepak bola berikut dengan gaya jurnalistik bahasa Indonesia yang rapi, informatif, dan natural seperti wartawan media olahraga profesional.

".
              "Syarat:
".
              "- Jangan menyalin kalimat mentah dari teks asli.
".
              "- Gunakan struktur paragraf yang enak dibaca.
".
              "- Hindari kalimat terlalu bertele-tele.
".
              "- Tetap pertahankan fakta utama, nama pemain, klub, skor, dan data penting.
".
              "- Gunakan sudut pandang pihak ketiga (bukan saya/kamu).

".
              "Judul asli: {$originalTitle}

".
              "Isi artikel asli:
{$originalContent}

".
              "Sekarang tulis ulang artikel lengkapnya:";

    $payload = [
        'contents' => [
            [
                'parts' => [
                    ['text' => $prompt]
                ]
            ]
        ],
        'generationConfig' => [
            'temperature' => floatval($template['temperature']),
            'maxOutputTokens' => intval($template['max_tokens']),
            'topP' => floatval($template['top_p']),
            'topK' => intval($template['top_k']),
        ]
    ];

    $headers = [
        'Content-Type: application/json'
    ];

    $response = http_post($endpoint, json_encode($payload), $headers);
    $json = json_decode($response, true);

    if (!isset($json['candidates'][0]['content']['parts'][0]['text'])) {
        throw new Exception('Response Gemini tidak sesuai format. Respon mentah: ' . $response);
    }

    return trim($json['candidates'][0]['content']['parts'][0]['text']);
}

/**
 * Proses batch artikel: ambil yang status = 'draft', rewrite, lalu simpan jadi 'processed'.
 */
function process_rewrite_batch($templateId, $limit = 3) {
    global $pdo;

    $stmtTemplate = $pdo->prepare("SELECT * FROM templates WHERE id = ?");
    $stmtTemplate->execute([$templateId]);
    $template = $stmtTemplate->fetch();
    if (!$template) {
        throw new Exception('Template tidak ditemukan (ID: ' . $templateId . ').');
    }

    $stmt = $pdo->prepare("SELECT * FROM articles WHERE status = 'draft' ORDER BY id ASC LIMIT ?");
    $stmt->bindValue(1, (int)$limit, PDO::PARAM_INT);
    $stmt->execute();
    $articles = $stmt->fetchAll();

    foreach ($articles as $article) {
        try {
            echo "Rewriting article ID {$article['id']}...\n";
            $rewritten = rewrite_with_gemini($template, $article['title'], $article['content']);

            $upd = $pdo->prepare("UPDATE articles SET rewritten = ?, status = 'processed' WHERE id = ?");
            $upd->execute([$rewritten, $article['id']]);

            echo "OK: ID {$article['id']} berhasil di-rewrite.\n";
        } catch (Exception $e) {
            echo "GAGAL rewrite ID {$article['id']}: " . $e->getMessage() . "\n";
        }
    }
}
